﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web.UI;
using System.Web.UI.WebControls;
using VeteransAffairs.Registries.Business;
using VeteransAffairs.Registries.BusinessManager;
using VeteransAffairs.Registries.BusinessManager.TBI;
using BusinessHelper = VeteransAffairs.Registries.Business.Utilities.Helpers;

public partial class TBI_Instruments_MedRed_L_Test : BaseTBIInstrumentPage {

    #region Variable Declarations

    PatientManager patientManager = new PatientManager();
    InstrumentType instrumentType = new InstrumentType();

    protected int _surveyTypeId = 7; // Database Id for MedRed L-Test 

    SURVEY _survey;   // the root object in the Survey set
    SURVEY_RESULT _surveyResult; // the object representing the actual data in the Survey

    List<STD_QUESTION> QuestionList = new List<STD_QUESTION>();

    #endregion

    /// <summary>
    /// Default Constructor
    /// </summary>
    public TBI_Instruments_MedRed_L_Test() {
        //Manager Save Notification Event Handler
        patientManager.BOSaveSuccess += manager_BOSaveSuccess;
    }

    #region Protected Methods

    /// <summary>
    /// Page Load method
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void Page_Load(object sender, EventArgs e) {
        TBIServiceTO _tbiServiceTo;

        try {
            // confirm instrument is active - if not, a redirect will occur.
            this.CheckActiveStatus(_surveyTypeId);

            _tbiServiceTo = (TBIServiceTO)Session["TBIServiceTO"];
            _tbiInstrumentManager.TbiServiceTo = _tbiServiceTo;

            string instTitle = _tbiInstrumentManager.GetInstrumentByID(_surveyTypeId).Name;

            LabelBreadCrumb.Text = "TBI Instruments > Instrument Associations > " + instTitle;
            LabelInstrumentTitle.Text = instTitle;
            Page.Title = "TBI Instruments - " + instTitle;

            if (_tbiInstrumentManager.TbiServiceTo != null) {
                Label labelUser = (Label)Master.FindControl("LabelCurrentUser");
                labelUser.Text = BusinessHelper.FixStringToPreventXss(_tbiInstrumentManager.TbiServiceTo.UserName);

                LabelPatient.Text = BusinessHelper.FixStringToPreventXss(_tbiInstrumentManager.TbiServiceTo.PatientName);
                LabelSSN.Text = BusinessHelper.FixStringToPreventXss(_tbiInstrumentManager.TbiServiceTo.PatientSnum);
            }
        } catch (Exception ex) {
            //Redirect to message that session is ended and need to log back in from CPRS.
            Exception _tbilogerror = HandleTBIException(ex, "MedRed_L-Test.aspx", "100027", "SESSIOENDED");
            Response.Redirect("Warning.aspx?err=2", false);
        }

        if (!IsPostBack) {
            if (instrumentType.Status != null) {
                _tbiInstrumentManager.TbiServiceTo.SelectedInstrumentType = instrumentType;
            }

            string _snum = "";
            if (_tbiInstrumentManager.TbiServiceTo != null) {
                _snum = _tbiInstrumentManager.TbiServiceTo.PatientSnum;
            }

            if (!string.IsNullOrEmpty(_snum))
            {
                string patICN = patientManager.GetPatientICNBySnum(_snum);

                if (!string.IsNullOrEmpty(patICN)) {
                    ViewState["PatientICN"] = patICN;
                    RefreshSurvey(patICN);
                } else {
                    throw new ArgumentException("Patient with this SSN does not exist.");
                }

                if (_survey != null && !_survey.isNew) {
                    SetUpSurvey(_survey);
                }
                string referrer = "";
                if (Request.UrlReferrer != null) {
                    referrer = Request.UrlReferrer.PathAndQuery;
                }
                ViewState["referrer"] = referrer;
            }

            //Check if the request is to start a new item 
            //a new item page request is signified when the querystring parameter id equals zero or is not present
            if (_tbiInstrumentManager.TbiServiceTo == null)
            {
                Logging.WriteLogToFile("MedRed_L-Test.Page_Load(): WarningRedirect: _tbiInstrumentManager.TbiServiceTo == null");
                //Redirect to message that session is ended and need to log back in from CPRS.
                Response.Redirect("Warning.aspx?err=2");
            } else {
                LabelPatient.Text = BusinessHelper.FixStringToPreventXss(_tbiInstrumentManager.TbiServiceTo.PatientName);
                LabelSSN.Text = BusinessHelper.FixStringToPreventXss(_tbiInstrumentManager.TbiServiceTo.PatientSnum.Trim().Length == 9 ? String.Format("{0:000-00-0000}", int.Parse(_tbiInstrumentManager.TbiServiceTo.PatientSnum)) : _tbiInstrumentManager.TbiServiceTo.PatientSnum);
            }
        }
        QuestionList = GetAllQuestionsForSurvey();

        // Set control values for Diagnosis Question
        ctrlDiagnosis.QuestionText = GetQuestionText("DX");
        ctrlDiagnosis.SurveyTypeCode = HiddenSurveyType.Value;

        LabelForDate.Text = GetQuestionText("11");
        LabelforLowerExtremityAmputation.Text = GetQuestionText("1");
        LabelForLowerExtremityAmputationLevel.Text = GetQuestionText("2");
        LabelforAssistiveDevice.Text = GetQuestionText("3");
        LabelForAssistiveDeviceSelection.Text = GetQuestionText("4");
        LabelPhysicalAssistance.Text = GetQuestionText("5");
        LabelTrial1Time.Text = GetQuestionText("6");
        LabelTrial1Distance.Text = GetQuestionText("7");
        LabelTrial2Time.Text = GetQuestionText("8");
        LabelTrial2Distance.Text = GetQuestionText("9");
        LabelUnstableTurning.Text = GetQuestionText("10");

        dpDate1.MaxDate = DateTime.Now;
    }

    protected override void LoadViewState(object savedState) {
        base.LoadViewState(savedState);
        string surveyXml = (string)ViewState["surveyXml"];
        _survey = SerializerUtil.Deserialize<SURVEY>(surveyXml);
    }

    protected override object SaveViewState() {
        ViewState["surveyXml"] = _survey.Serialize();
        return base.SaveViewState();
    }

    /// <summary>
    /// Returns the Question_Text of the question with the passed Question_Number field
    /// </summary>
    /// <param name="questionId"></param>
    /// <returns></returns>
    protected string GetQuestionText(string questionId) {
        foreach (STD_QUESTION ques in QuestionList) {
            if (ques.QUESTION_NUMBER.ToLower() == questionId.ToLower()) {
                return ques.QUESTION_TEXT;
            }
        }

        return "";
    }

    /// <summary>
    /// Returns the Question_Id of the question with the passed Question_Number field
    /// </summary>
    /// <param name="questionId"></param>
    /// <returns></returns>
    protected int GetQuestionId(string questionNumber) {
        foreach (STD_QUESTION ques in QuestionList) {
            if (ques.QUESTION_NUMBER.ToLower() == questionNumber.ToLower()) {
                return ques.ID;
            }
        }

        return 0;
    }

    /// <summary>
    /// Fired when Lower Extremity Amputation selection changed.
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void rblLowerExtremityAmputation_SelectedIndexChanged(object sender, EventArgs e) {
        if (rblLowerExtremityAmputation.SelectedItem.Text.ToLower() == "yes") {
            pnlLowerExtremityAmputationLevel.Visible = true;
        } else {
            pnlLowerExtremityAmputationLevel.Visible = false;
            rblLowerExtremityAmputationLevel.SelectedIndex = -1;
        }
    }

    /// <summary>
    /// Fired when the Assistive Device Selection changes.
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void rblAssistiveDevice_SelectedIndexChanged(object sender, EventArgs e) {
        if (rblAssistiveDevice.SelectedItem.Text.ToLower() == "yes") {
            pnlAssistiveDeviceSelection.Visible = true;
        } else {
            pnlAssistiveDeviceSelection.Visible = false;
            rblAssistiveDeviceSelection.SelectedIndex = -1;
        }
    }

    /// <summary>
    /// Save Draft button clicked.
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    public void btnSaveSurveyDraft_Click(object sender, EventArgs e) {
        Int32 status = 0;
        SaveFailMessage = "Save Failed. Please press " + btnSaveSurveyDraft.Text + " button again to avoid data loss.";
        SaveCompleteSurvey(ref status);
        SetUpSurvey(_survey);
    }

    /// <summary>
    /// Save and Prepare Note button clicked.
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void btnSubmit_Click(object sender, EventArgs e) {
        Int32 status = 0;
        SaveFailMessage = "Save Failed. Please press " + btnSubmitSurvey.Text + " button again to avoid data loss.";
        SaveCompleteSurvey(ref status);

        if (status > 0) {
            Session["survey"] = _survey;
            //Session["rawScore"] = hiddenRawScore.Value.ToString();
            //Session["tScore"] = hiddenTScore.Value.ToString();
            Response.Redirect("InstrumentSummaryNote.aspx", false);
        } else {
            SetUpSurvey(_survey); //Follow Save Draft logic 
        }
    }

    protected void InActiveTimer_Tick(object sender, EventArgs e) {
        Int32 status = 0;
        SaveFailMessage = "Save Failed. Please press " + btnSaveSurveyDraft.Text + " button again to avoid data loss.";

        Page.Validate();

        if (Page.IsValid) {
            SaveCompleteSurvey(ref status);
            //SetUpSurvey(_survey);
            Response.Redirect("MedRed_L-Test.aspx", false);
        } else {
            SaveFailMessage = "Auto-Save Validation Failed. Please complete required fields in the survey and press " + btnSaveSurveyDraft.Text + " button again to avoid data loss.";
        }
    }

    protected void btnCancel_Click(object sender, EventArgs e) {
        ViewState["surveyXml"] = null;
        ViewState["PatientICN"] = null;
        ViewState["referrer"] = null;
        Session.Remove("survey");
        Response.Redirect("Instrument.aspx", false);
        //this.ClientScript.RegisterStartupScript(typeof(Page), "closePage", "<script type='text/javascript'> window.open('','_self','');window.close(); </script>");
    }

    /// <summary>
    /// Hide or Show instructions
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void lnkBtnToggleInstructions_OnClick(object sender, EventArgs e) {
        if (PanelInstructions.Visible) {
            PanelInstructions.Visible = false;
            lnkBtnToggleInstructions.Text = "Click for Instructions";
        } else {
            PanelInstructions.Visible = true;
            lnkBtnToggleInstructions.Text = "Hide Instructions";
        }
    }

    #endregion

    #region Private Methods

    /// <summary>
    /// Get all questions for this survey
    /// </summary>
    /// <returns></returns>
    private List<STD_QUESTION> GetAllQuestionsForSurvey() {
        return _tbiInstrumentManager.SelectAllQuestions(_surveyTypeId).ToList();
    }

    /// <summary>
    /// Refresh Survey
    /// </summary>
    /// <param name="patientICN"></param>
    private void RefreshSurvey(string patientICN) {
        var surveys = _tbiInstrumentManager.SelectByPatientICNAndSurveyType(patientICN, _surveyTypeId, true).ToList();

        if (surveys.Count > 0) {
            _survey = surveys.First();
        } else {
            _survey = new SURVEY();
        }

        _survey.SetAsChangeTrackingRoot();
    }

    /// <summary>
    /// Save Complete Survey
    /// </summary>
    /// <param name="status"></param>
    private void SaveCompleteSurvey(ref Int32 status) {
        Page.Validate();
        if (Page.IsValid) {

            string patientICN = String.Empty;

            status = 0;

            //Retrieve patient's ICN from the session variable
            if (ViewState["PatientICN"] != null) {
                patientICN = ViewState["PatientICN"].ToString();
                if (String.IsNullOrEmpty(patientICN.Trim()))
                {
                    Logging.WriteLogToFile("MedRed_L-Test.SaveCompletedSurvey(): WarningRedirect: String.IsNullOrEmpty(patientICN.Trim()");
                    //Lost state. Redirect to message that need to log back in from CPRS.
                    Response.Redirect("Warning.aspx?err=2", false);
                }
            }
            else
            {
                Logging.WriteLogToFile("MedRed_L-Test.SaveCompletedSurvey(): WarningRedirect: ViewState[PatientICN] != null");
                //Lost state. Redirect to message that need to log back in from CPRS.
                Response.Redirect("Warning.aspx?err=2");
            }

            if (_survey != null) {
                if (_survey.isNew) {
                    //Ensure that a concurrent session hasn't already inserted a record for this survey
                    var surveys = _tbiInstrumentManager.SelectByPatientICNAndSurveyType(patientICN, _surveyTypeId, true).ToList();

                    if (surveys.Count > 0)
                    {
                        Logging.WriteLogToFile("MedRed_L-Test.SaveCompletedSurvey(): WarningRedirect: surveys.Count > 0");
                        //A concurrent session has already inserted a record for this survey and it is not new anymore.
                        Response.Redirect("Warning.aspx?err=9", false);
                    } else {
                        _survey.SURVEY_STATUS = "inprocess";

                        _survey.SetAsInsertOnSubmit();

                        _survey.patient_ICN = patientICN;

                        _survey.STD_SURVEY_TYPE_ID = _surveyTypeId;

                        // TODO: change to datepicker
                        _survey.SURVEY_DATE = DateTime.Today;

                        //Save survey if it's new before adding any responses
                        SuppressSaveMessage = true;
                        _tbiInstrumentManager.Update(_survey);
                        SuppressSaveMessage = false;

                        RefreshSurvey(patientICN);
                    }
                }

                SaveSurvey(ref _survey);

                _survey.SetAsUpdateOnSubmit();
                status = _tbiInstrumentManager.Update(_survey);

                ((RegistryInstrumentMaster)Master).SetDirtyFlag(false);

                RefreshSurvey(patientICN);
                SuppressSaveMessage = false;
            }
        }
    }

    /// <summary>
    /// Save survey.
    /// </summary>
    /// <param name="survey"></param>
    private void SaveSurvey(ref SURVEY survey) {
        List<STD_QUESTION> questions = GetAllQuestionsForSurvey();

        //Delete all the survey results from database if there are any.
        SURVEY surveyWithResults = _tbiInstrumentManager.GetSurveyWithAllAnswers(survey.SURVEYS_ID);
        _tbiInstrumentManager.DeleteAllSurveyResults(surveyWithResults);

        // Question DX
        _surveyResult = new SURVEY_RESULT();

        try {
            // Add Question DX choice
            _surveyResult.STD_QUESTION_CHOICE_ID = Convert.ToInt32(ctrlDiagnosis.GetSelectedId());
            _surveyResult.RESULT_TEXT = ctrlDiagnosis.GetSelectedResultText();
            survey.SURVEY_RESULTs.Add(_surveyResult);
        } catch (Exception surveyEx) {
            Exception _tbilogerror = HandleTBIException(surveyEx, "MedRed_L-Test.aspx", "100027", "SaveSurvey() - Question DX");
        }

        // Question 1 - Lower extremity amputation
        STD_QUESTION question1 = questions.FirstOrDefault(Q => Q.QUESTION_NUMBER == "1");
        _surveyResult = new SURVEY_RESULT();

        try {
            // Add Question 1 choice
            _surveyResult.STD_QUESTION_CHOICE_ID = Convert.ToInt32(rblLowerExtremityAmputation.SelectedValue);
            _surveyResult.RESULT_TEXT = rblLowerExtremityAmputation.SelectedItem.Text;
            survey.SURVEY_RESULTs.Add(_surveyResult);
        } catch (Exception surveyEx) {
            Exception _tbilogerror = HandleTBIException(surveyEx, "MedRed_L-Test.aspx", "100027", "SaveSurvey() - Question1");
        }

        // Question 2 - Lower Extremity Amputation Selection
        if (rblLowerExtremityAmputationLevel.SelectedIndex > -1) {
            STD_QUESTION question2 = questions.FirstOrDefault(Q => Q.QUESTION_NUMBER == "2");
            _surveyResult = new SURVEY_RESULT();

            try {
                // Add Question 2 choice
                _surveyResult.STD_QUESTION_CHOICE_ID = Convert.ToInt32(rblLowerExtremityAmputationLevel.SelectedValue);
                _surveyResult.RESULT_TEXT = rblLowerExtremityAmputationLevel.SelectedItem.Text;
                survey.SURVEY_RESULTs.Add(_surveyResult);
            } catch (Exception surveyEx) {
                Exception _tbilogerror = HandleTBIException(surveyEx, "MedRed_L-Test.aspx", "100027", "SaveSurvey() - Question2");
            }
        }

        // Question 3 - Assistive Device
        STD_QUESTION question3 = questions.FirstOrDefault(Q => Q.QUESTION_NUMBER == "3");
        _surveyResult = new SURVEY_RESULT();

        try {
            // Add Question 3 choice
            _surveyResult.STD_QUESTION_CHOICE_ID = Convert.ToInt32(rblAssistiveDevice.SelectedValue);
            _surveyResult.RESULT_TEXT = rblAssistiveDevice.SelectedItem.Text;
            survey.SURVEY_RESULTs.Add(_surveyResult);
        } catch (Exception surveyEx) {
            Exception _tbilogerror = HandleTBIException(surveyEx, "MedRed_L-Test.aspx", "100027", "SaveSurvey() - Question3");
        }

        // Question 4 - Assistive Device Selection
        if (rblAssistiveDeviceSelection.SelectedIndex > -1) {
            STD_QUESTION question4 = questions.FirstOrDefault(Q => Q.QUESTION_NUMBER == "4");
            _surveyResult = new SURVEY_RESULT();

            try {
                // Add Question 4 choice
                _surveyResult.STD_QUESTION_CHOICE_ID = Convert.ToInt32(rblAssistiveDeviceSelection.SelectedValue);
                _surveyResult.RESULT_TEXT = rblAssistiveDeviceSelection.SelectedItem.Text;
                survey.SURVEY_RESULTs.Add(_surveyResult);
            } catch (Exception surveyEx) {
                Exception _tbilogerror = HandleTBIException(surveyEx, "MedRed_L-Test.aspx", "100027", "SaveSurvey() - Question4");
            }
        }

        // Question 5 - Physical Assistance
        STD_QUESTION question5 = questions.FirstOrDefault(Q => Q.QUESTION_NUMBER == "5");
        _surveyResult = new SURVEY_RESULT();

        try {
            // Add Question 5 choice
            _surveyResult.STD_QUESTION_CHOICE_ID = Convert.ToInt32(rblPhysicalAssistance.SelectedValue);
            _surveyResult.RESULT_TEXT = rblPhysicalAssistance.SelectedItem.Text;
            survey.SURVEY_RESULTs.Add(_surveyResult);
        } catch (Exception surveyEx) {
            Exception _tbilogerror = HandleTBIException(surveyEx, "MedRed_L-Test.aspx", "100027", "SaveSurvey() - Question5");
        }

        // Question 6 - Trial 1 Time
        STD_QUESTION question6 = questions.FirstOrDefault(Q => Q.QUESTION_NUMBER == "6");
        _surveyResult = new SURVEY_RESULT();

        try {
            // Add Question 6 choice
            _surveyResult.STD_QUESTION_CHOICE_ID = question6.STD_QUESTION_CHOICEs.FirstOrDefault().STD_QUESTION_CHOICE_ID;
            _surveyResult.RESULT_TEXT = txtTrial1Time.Text;
            survey.SURVEY_RESULTs.Add(_surveyResult);
        } catch (Exception surveyEx) {
            Exception _tbilogerror = HandleTBIException(surveyEx, "MedRed_L-Test.aspx", "100027", "SaveSurvey() - Question6");
        }

        // Question 7 - Trial 1 Distance
        STD_QUESTION question7 = questions.FirstOrDefault(Q => Q.QUESTION_NUMBER == "7");
        _surveyResult = new SURVEY_RESULT();

        try {
            // Add Question 7 choice
            _surveyResult.STD_QUESTION_CHOICE_ID = question7.STD_QUESTION_CHOICEs.FirstOrDefault().STD_QUESTION_CHOICE_ID;
            _surveyResult.RESULT_TEXT = txtTrial1Distance.Text;
            survey.SURVEY_RESULTs.Add(_surveyResult);
        } catch (Exception surveyEx) {
            Exception _tbilogerror = HandleTBIException(surveyEx, "MedRed_L-Test.aspx", "100027", "SaveSurvey() - Question7");
        }

        // Question 8 - Trial 2 Time
        if (!String.IsNullOrEmpty(txtTrial2Time.Text)) {
            STD_QUESTION question8 = questions.FirstOrDefault(Q => Q.QUESTION_NUMBER == "8");
            _surveyResult = new SURVEY_RESULT();

            try {
                // Add Question 8 choice
                _surveyResult.STD_QUESTION_CHOICE_ID = question8.STD_QUESTION_CHOICEs.FirstOrDefault().STD_QUESTION_CHOICE_ID;
                _surveyResult.RESULT_TEXT = txtTrial2Time.Text;
                survey.SURVEY_RESULTs.Add(_surveyResult);
            } catch (Exception surveyEx) {
                Exception _tbilogerror = HandleTBIException(surveyEx, "MedRed_L-Test.aspx", "100027", "SaveSurvey() - Question8");
            }
        }

        // Question 9 - Trial 2 Distance
        if (!String.IsNullOrEmpty(txtTrial2Distance.Text) && !String.IsNullOrEmpty(txtTrial2Time.Text)) {
            STD_QUESTION question9 = questions.FirstOrDefault(Q => Q.QUESTION_NUMBER == "9");
            _surveyResult = new SURVEY_RESULT();

            try {
                // Add Question 9 choice
                _surveyResult.STD_QUESTION_CHOICE_ID = question9.STD_QUESTION_CHOICEs.FirstOrDefault().STD_QUESTION_CHOICE_ID;
                _surveyResult.RESULT_TEXT = txtTrial2Distance.Text;
                survey.SURVEY_RESULTs.Add(_surveyResult);
            } catch (Exception surveyEx) {
                Exception _tbilogerror = HandleTBIException(surveyEx, "MedRed_L-Test.aspx", "100027", "SaveSurvey() - Question9");
            }
        }

        // Question 10 - Unstable Turning
        STD_QUESTION question10 = questions.FirstOrDefault(Q => Q.QUESTION_NUMBER == "10");
        _surveyResult = new SURVEY_RESULT();

        try {
            // Add Question 10 choice
            _surveyResult.STD_QUESTION_CHOICE_ID = Convert.ToInt32(rblUnstableTurning.SelectedValue);
            _surveyResult.RESULT_TEXT = rblUnstableTurning.SelectedItem.Text;
            survey.SURVEY_RESULTs.Add(_surveyResult);
        } catch (Exception surveyEx) {
            Exception _tbilogerror = HandleTBIException(surveyEx, "MedRed_L-Test.aspx", "100027", "SaveSurvey() - Question10");
        }

        // Question 11 - Date
        if (dpDate1.SelectedDate != null) {
            STD_QUESTION question11 = questions.FirstOrDefault(Q => Q.QUESTION_NUMBER == "11");
            _surveyResult = new SURVEY_RESULT();

            try {
                // Add Question 11 choice
                _surveyResult.STD_QUESTION_CHOICE_ID = question11.STD_QUESTION_CHOICEs.FirstOrDefault().STD_QUESTION_CHOICE_ID;
                _surveyResult.RESULT_TEXT = ((DateTime)dpDate1.SelectedDate).ToShortDateString();
                survey.SURVEY_RESULTs.Add(_surveyResult);
            } catch (Exception surveyEx) {
                Exception _tbilogerror = HandleTBIException(surveyEx, "MedRed_L-Test.aspx", "100027", "SaveSurvey() - Question11");
            }
        }

        // determine if multiple tests were done
        if (!String.IsNullOrEmpty(txtTrial2Time.Text) && !String.IsNullOrEmpty(txtTrial2Distance.Text)) {
            int time1 = Convert.ToInt32(txtTrial1Time.Text);
            int distance1 = Convert.ToInt32(txtTrial1Distance.Text);

            int time2 = Convert.ToInt32(txtTrial2Time.Text);
            int distance2 = Convert.ToInt32(txtTrial2Distance.Text);

            decimal averageTime = ((time1 + time2) / 2);
            decimal averageDist = ((distance1 + distance2) / 2);

            HiddenAverageTime.Value = averageTime.ToString();
            HiddenAverageDistance.Value = averageDist.ToString();

            // Question 12 - Average Time
            STD_QUESTION question12 = questions.FirstOrDefault(Q => Q.QUESTION_NUMBER == "12");
            _surveyResult = new SURVEY_RESULT();

            try {
                // Add Question 12 choice
                _surveyResult.STD_QUESTION_CHOICE_ID = question12.STD_QUESTION_CHOICEs.FirstOrDefault().STD_QUESTION_CHOICE_ID;
                _surveyResult.RESULT_TEXT = averageTime.ToString();
                survey.SURVEY_RESULTs.Add(_surveyResult);
            } catch (Exception surveyEx) {
                Exception _tbilogerror = HandleTBIException(surveyEx, "MedRed_L-Test.aspx", "100027", "SaveSurvey() - Question12");
            }

            // Question 13 - Average Distance
            STD_QUESTION question13 = questions.FirstOrDefault(Q => Q.QUESTION_NUMBER == "13");
            _surveyResult = new SURVEY_RESULT();

            try {
                // Add Question 13 choice
                _surveyResult.STD_QUESTION_CHOICE_ID = question13.STD_QUESTION_CHOICEs.FirstOrDefault().STD_QUESTION_CHOICE_ID;
                _surveyResult.RESULT_TEXT = averageDist.ToString();
                survey.SURVEY_RESULTs.Add(_surveyResult);
            } catch (Exception surveyEx) {
                Exception _tbilogerror = HandleTBIException(surveyEx, "MedRed_L-Test.aspx", "100027", "SaveSurvey() - Question13");
            }
        }
    }// END SaveSurvey

    /// <summary>
    /// Initialize/setup controls
    /// </summary>
    /// <param name="survey"></param>
    private void SetUpSurvey(SURVEY survey) {
        try {
            foreach (var sResult in survey.SURVEY_RESULTs) {
                switch (sResult.STD_QUESTION_CHOICE.STD_QUESTION.QUESTION_NUMBER) {
                    case "DX":
                        if (sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID > -1) {
                            ctrlDiagnosis.SelectedChoiceId = sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID.ToString();

                            if (sResult.STD_QUESTION_CHOICE.CHOICE_TEXT.ToLower().Contains("other")) {
                                ctrlDiagnosis.SetOtherDescriptionText(sResult.RESULT_TEXT);
                            }
                        } else {
                            ctrlDiagnosis.SelectedChoiceId = null;
                        }
                        break;
                    case "1":
                        if (sResult.STD_QUESTION_CHOICE.CHOICE_SORT_ORDER.HasValue) {
                            rblLowerExtremityAmputation.SelectedValue = sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID.ToString();
                        } else {
                            rblLowerExtremityAmputation.SelectedIndex = -1;
                        }

                        break;
                    case "2":
                        pnlLowerExtremityAmputationLevel.Visible = true;
                        if (sResult.STD_QUESTION_CHOICE.CHOICE_SORT_ORDER.HasValue) {
                            rblLowerExtremityAmputationLevel.SelectedValue = sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID.ToString();
                        } else {
                            rblLowerExtremityAmputationLevel.SelectedIndex = -1;
                        }

                        break;
                    case "3":
                        if (sResult.STD_QUESTION_CHOICE.CHOICE_SORT_ORDER.HasValue) {
                            rblAssistiveDevice.SelectedValue = sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID.ToString();
                        } else {
                            rblAssistiveDevice.SelectedIndex = -1;
                        }

                        break;
                    case "4":
                        pnlAssistiveDeviceSelection.Visible = true;
                        if (sResult.STD_QUESTION_CHOICE.CHOICE_SORT_ORDER.HasValue) {
                            rblAssistiveDeviceSelection.SelectedValue = sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID.ToString();
                        } else {
                            rblAssistiveDeviceSelection.SelectedIndex = -1;
                        }

                        break;
                    case "5":
                        if (sResult.STD_QUESTION_CHOICE.CHOICE_SORT_ORDER.HasValue) {
                            rblPhysicalAssistance.SelectedValue = sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID.ToString();
                        } else {
                            rblPhysicalAssistance.SelectedIndex = -1;
                        }

                        break;
                    case "6":
                        if (!String.IsNullOrEmpty(sResult.RESULT_TEXT)) {
                            txtTrial1Time.Text = sResult.RESULT_TEXT;
                        }

                        break;
                    case "7":
                        if (!String.IsNullOrEmpty(sResult.RESULT_TEXT)) {
                            txtTrial1Distance.Text = sResult.RESULT_TEXT;
                        }
                        break;
                    case "8":
                        if (!String.IsNullOrEmpty(sResult.RESULT_TEXT)) {
                            txtTrial2Time.Text = sResult.RESULT_TEXT;
                        }
                        break;
                    case "9":
                        if (!String.IsNullOrEmpty(sResult.RESULT_TEXT)) {
                            txtTrial2Distance.Text = sResult.RESULT_TEXT;
                        }
                        break;
                    case "10":
                        if (sResult.STD_QUESTION_CHOICE.CHOICE_SORT_ORDER.HasValue) {
                            rblUnstableTurning.SelectedValue = sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID.ToString();
                        } else {
                            rblUnstableTurning.SelectedIndex = -1;
                        }
                        break;
                    case "11":
                        if (!String.IsNullOrEmpty(sResult.RESULT_TEXT)) {
                            DateTime temp;
                            if (DateTime.TryParse(sResult.RESULT_TEXT, out temp)) {
                                dpDate1.SelectedDate = temp;
                            }
                        }
                        break;
                    case "12":
                        // Average Time:
                        if (!String.IsNullOrEmpty(sResult.RESULT_TEXT)) {
                            HiddenAverageTime.Value = sResult.RESULT_TEXT;
                        }
                        break;
                    case "13":
                        // Average Distance
                        if (!String.IsNullOrEmpty(sResult.RESULT_TEXT)) {
                            HiddenAverageDistance.Value = sResult.RESULT_TEXT;
                        }
                        break;
                    default:
                        break;
                }
            }
        } catch (Exception ex) {
            Exception _tbilogerror = HandleTBIException(ex, "MedRed_L-Test.aspx", "100027", "SetUpSurvey()");
        }
    }// END SetUpSurvey

    #endregion
}